#| 
 | Name: Omer Bar-or
 |
 | Please note the comment in section "Other Pruning Events" about the
 | conversation I had with Sam Luckenbill that affected my program.
 |#

; (prove-contradiction clauses): given clauses (a list of logical clauses
; in clause form) this function returns either a tree showing how it
; derived a contradiction (using resolution), or "FAIL," meaning that no
; contradiction could be found.
; 
; Clause form, in this context, is a list of literals which is treated as
; a clause with the disjunction of each of those literals. A literal is
; either any symbol or the negation of any symbol, where negation is
; shown with a two-element list, the first element of which is the word
; "not" and the second of which is the negated symbol.
; 
; A derivation tree is a tree, the internal nodes of which are
; three-element lists, and the leaf nodes of which are single-element
; lists. The leaves are clauses from the input, and each internal node is
; of the form (c r1 r2) where c is the clause this node represents and r1
; and r2 are the clauses that resolve into c.
;
; This function works by first doing preprocessing on clauses,
; then calling prove-aux, which derives a contradiction if one exists, and
; finally doing post-processing to get the solution in the requested form.
; The preprocessing done is with functions build-ds and add-sort. build-ds
; turns the clauses into a clause nodes for prove-aux, and add-sort sorts
; them by their f-cost (see prove-aux for a description of f-cost).
; prove-aux treats the clauses as a priority queue, running through each
; one, resolving it with the rest, and inserting the results back onto
; the queue. And, it stops when a contradiciton is found.
; The postprocessing done is to first reverse the solution from prove-aux
; so that the contradicting node (with a nil clause) is first in the list,
; then checking if that node actually is nil (if not, we failed to find a
; contradiction). Then, it sorts this solution by the number of steps
; required to reach that step. It does this so that, when build-sol builds
; the solution, it never needs to backtrack (see build-sol).
(defun prove-contradiction (clauses)
	(cond ((null clauses) (list ())) ; just in case our list is empty...
				(t (let ((pre_sol (reverse (prove-aux (add-sort (build-ds clauses 0)) ; get prove-aux's solution
																							(length clauses)))))
						 (cond ((not (null (third (car pre_sol)))) 'FAIL) ; if prove-aux failed
									 ((null pre_sol) 'FAIL) ; case in which build-ds determined all clauses as vacuously true
									 (t (build-sol (cons (car pre_sol) ; build a solution
																			 (reverse-car-sort (cdr pre_sol))) ; sort cdr of list for ease of building
																 (cadr (car pre_sol))))))))) ; root is first element in list

; (prove-aux clauses n): given clauses (a list of clause nodes (see
; build-ds)) with ids less than n, this function adds to clauses using
; resolution until it finds a contradiction. It uses n to number the nodes
; it adds. It returns a list of clause nodes including the contradiction
; and any other node that could possibly have led to that contradiction,
; with the contradiction itself always as the last element of the list. If
; no contradiction is found, it returns all clauses it found.
; The list of clauses is a priority queue, which prioritizes clauses that
; are more likely to be optimal solutions. The way it does this is similar
; to A* tree search: it adds the depth of the tree derivation to this node
; (g(x)) with the number of elements left to resolve in the clause (h(x)).
; Because only one element can be resolved at a time, h(x) is an admissible
; heuristic to predict solution depth from this node. Hence, organizing our
; list of clauses according to f(x)=g(x)+h(x) ensures that when we choose a
; node with h(x)=0 for expansion (a contradiction), it is the end of the
; optimal path to a contradiction.
; To ease post-production (see prove-contradiction), this function drops
; from its final solution any clause that is obviously not part of the
; solution, that is, any node whose f-cost is greater than the g-cost to
; our solution (because, obviously, we didn't use this node in any
; resolution).
(defun prove-aux (clauses n)
	(cond ((null clauses) nil) ; No more clauses to analyze... Return.
				((null (third (car clauses))) ; Contradiction!
				 (append (drop-big (cdr clauses) (car (car clauses))) ; drop unnecessary nodes
								 (list (car clauses)))) ; make sure contradiction is last
				(t (let ((addition (one-step-resolve (car clauses) ; resolve car
																						 (cdr clauses) ; with all of cdr
																						 n)))
						 (cons (car clauses) ; This node was expanded, so it might be in the solution
									 (prove-aux (remove-caddr-dups ; prune clauses by removing duplicates
																(merge-nodes (cdr clauses) ; merge current clause list
																						 (add-sort addition))) ; with the new clauses
															(+ n (length addition)))))))) ; create new n

; (build-ds clauses n): given a list of clauses in clause form (see
; prove-contradiction) and a node id, this function returns the a list of
; clause nodes that matches the clauses, where each node is numbered
; incrementally, starting with n.
;
; The data structure has two forms, for leaf nodes and for internal nodes.
; Leaf nodes look like: (g-cost id clause)
; Internal nodes look like: (g-cost id clause rid1 rid2)
; Where...
; g-cost: number of resolutions required to reach this node (see prove-aux)
; id: this node's id (a unique number associated with the node)
; clause: the clause this node represents
; rid1: the id of one node used in the resolution to create this node.
; rid2: the id of the other node used to create this node.
;
; We use this data structure because it is concise enough to let us add
; many nodes to our list (see prove-aux) before running out of space, but
; it retains all of the information we need to construct a solution out of
; a list of nodes (see build-sol), which means that we don't have to search
; the nodes in a depth-first fashion to construct the path to the node.
;
; Note: all clauses passed to build-ds are, by definition, leaf clauses,
; so each has a g-cost of 0 and only three elements.
;
; Note 2: just in case we have bad input, we run has-negate and remove-dups
; on all clauses before bothering to add them to our list.
(defun build-ds (clauses n)
	(cond ((null clauses) nil) ; no more clauses to build
				((has-negate (car clauses)) (build-ds (cdr clauses) (+ n 1))) ; bad clause, only add the rest
				(t (cons (list 0 n (remove-dups (car clauses))) ; build the node, removing duplicate literals
								 (build-ds (cdr clauses) (+ n 1)))))) ; add the rest

; (build-sol clauses id): given a list of clause nodes, sorted in reverse
; order by g-cost, and the id of some node in that list, this function
; returns a tree of the kind requested by the solution (see
; prove-contradiction). It runs recursively by first finding the node,
; and if the node is a leaf, simply returning the clause of that node as
; the only node in a tree. If, on the other hand, the node is an internal
; node, the function returns a tree with the clause as a root, and the
; results of running recursively on each of the nodes used to build
; this one as its two children.
; Because the nodes are stored in reverse order by g-cost, when running
; recursively, we are guaranteed that the parents are located later in the
; list than this node, because their cost has to be at most one less than
; than the cost to reach this node.
(defun build-sol (clauses id)
	(cond ((not (= (cadr (car clauses)) id)) (build-sol (cdr clauses) id)) ; wrong node, keep going)
				((= (length (car clauses)) 3) (list (third (car clauses)))) ; leaf node
				(t (list (third (car clauses)) ; internal node, built tree with the clause
								 (build-sol (cdr clauses) (fourth (car clauses))) ; first child
								 (build-sol (cdr clauses) (fifth (car clauses))))))) ; second child
				 

#| ##### Resolution Functions ##### |#

; (one-step-resolve v l n): given a single node v, a list of other nodes l,
; and a minimum id n, this function returns the list of nodes that result
; from applying resolution between v and each element in l, where the
; resulting nodes are numbered incrementally from n.
; It works by using the lit-resolve function to compute a single
; resolution, and if that resolution succeeded, building a node for it.
(defun one-step-resolve (v l n)
	(cond ((null l) ()) ; no more nodes to resolve
				(t (let ((res (lit-resolve (third v) ; resolve with v's clause...
																		(third (car l)) ; ... car l's clause...
																		(third v)))) ; ... and v's clause...
						 (cond ((null (car res)) (one-step-resolve v (cdr l) n)) ; failed
									 (t (cons (list (+ 1 (max (car v) (car (car l)))) ; succeeded: build new node
																	n ; id
																	(remove-dups (cadr res)) ; remove duplicate literals from the clause
																	(cadr v) ; rid1
																	(cadr (car l))) ; rid2
														(one-step-resolve v (cdr l) (+ n 1))))))))) ; continue

; (lit-resolve a b orig_a): given two clauses b and orig_a, and clause a,
; which is a subset of orig_a, this function returns a two-element list.
; The first element is either t or nil, depending on whether applying
; resolution was successful or not, and the second element is the result
; of the resolution, that is, every element in b and orig_a except for
; the resolved literal from each. It works by recursively running through
; the list a, comparing it to the first element in b, and when a runs out,
; running recursively on (orig_a (cdr b) orig_a), that is, resetting a and
; moving on to the next element in b. If it finds a match, it checks if
; there are any _more_ matches. If there are, then the resolution leads to
; a vacuous truth (because the resolution will have something like A or
; NOT A or ..., which is always true), and we can throw it out as useless
; (see has-negate). If there are no more matches, it returns a list with
; all elements in b except for the match and all elements in orig_a except
; for the match. Note that because we iterate through b, if we run
; recursively on (cdr b), we have to re-add (car b) to the solution, since
; it is a unique literal in b.
(defun lit-resolve (a b orig_a)
	(cond ((null b) (list nil nil)) ; nothing more to resolve, return failure
				((null a) (let ((next (lit-resolve orig_a (cdr b) orig_a))) ; finished one iteration of a
										(list (car next) (cons (car b) (cadr next))))) ; add (car b) after recursion
				((negate (car a) (car b)) ; a negates b!
				 (cond ((has-negate (append (remove-list (car a) orig_a) (cdr b))) ; any more negations?
								(list nil nil)) ; if so, this is a vacuous truth; fail.
							 (t (list t (append (remove-list (car a) orig_a) ; otherwise, success!
																						 (cdr b))))))
				(t (lit-resolve (cdr a) b orig_a)))) ; iterate on a...

; (negate a b): given two literals, a and b, return whether a and b are
; negating literals. If they are, one must be an atom, and the other must
; be a two-element list, with the first element as "not" and the second
; equal to the atom of the other. The function checks which one is an
; atom (if either) and if the other is a two element list matching the
; requirements.
(defun negate (a b)
	(cond ((atom a) ; a is an atom
				 (cond ((atom b) nil) ; if b is an atom too, can't be negations
							 (t (and (equal (car b) 'not) (equal (cadr b) a))))) ; b = (not a)?
				((equal (car a) 'not) (equal b (cadr a))) ; a = (not b)?
				(t nil))) ; none of the above

#| ##### Removing Duplicates ##### |#

; (remove-dups l): given a list l, this function returns a list without
; any duplicates. In the context of this program, it runs on clauses to
; make sure that a literal doesn't appear in the clause twice. (If it does,
; even if it gets removed via resolution, it will still show up, which is
; not what we want.) It runs recursively on each element on the list,
; removing anything equivalent in the rest of the list before recurring.
(defun remove-dups (l)
	(cond ((null l) nil)
				(t (cons (car l) (remove-dups (remove-list (car l) (cdr l)))))))

; (remove-list v l): given a value v and a list l, return l with all
; instances of v removed. This function looks a lot like the built-in
; remove, but because it uses the "equal" predicate, v can be a list
; (e.g., "(not a)").
(defun remove-list (v l)
	(cond ((null l) nil)
				((equal v (car l)) (remove-list v (cdr l)))
				(t (cons (car l) (remove-list v (cdr l))))))

; (remove-caddr-dups l): given a list l, this funtion returns l without any
; duplicates, where duplicates are elements whose caddr (third element) are
; equivalent. It and remove-caddr below look identical to remove-dups and
; remove-list above, except for the equivalency check, which is handled
; by list-equal.
; Note: though this function works generally on a list of lists with at
; least three elements (where the third element is another list), it is
; used specifically to prune out equivalent elements in the priority queue
; in prove-aux. Because the clauses of the nodes in that queue can't easily
; be sorted, the equivalency check checks to see if both elements are
; subsets of each other. Finally, because l is a priority queue, if we
; find something equivalent to (car l) later in the list, it cannot
; possibly be a better node than (car l). They must have the same h-cost
; because they are identical, and (car l) has a better or equal f-cost, so
; it must also have a better or equal g-cost. Therefore, we should always
; keep (car l) and remove the other duplicate. We need not compare the 
; costs of the two to see which will lead us to a more optimal solution.
(defun remove-caddr-dups (l)
	(cond ((null l) nil)
				(t (cons (car l) (remove-caddr-dups (remove-caddr (car l) (cdr l)))))))

; (remove-caddr v l): given a value v and a list l, where v is a list of
; length at least three (with the third element as another list) and l
; a list of values with the same properties, return a list equivalent to l
; but with each element whose third element is equivalent to v's third
; element removed.
(defun remove-caddr (v l)
	(cond ((null l) nil)
				((list-equal (third v) (third (car l)))
				 (remove-caddr v (cdr l)))
				(t (cons (car l) (remove-caddr v (cdr l))))))

; (list-equal l1 l2): given two lists, return whether the lists have
; identical elements, regardless of the order of those elements. It
; just checks if each is a subset of the other.
(defun list-equal (l1 l2)
	(and (list-subset l1 l2) (list-subset l2 l1)))

; (list-subset l1 l2): given two lists l1 and l2, return whether l1 is a
; subset of l2, i.e., if each member of l1 is in l2.
(defun list-subset (l1 l2)
	(cond ((null l1) t)
				((list-member (car l1) l2) (list-subset (cdr l1) l2))
				(t nil)))

; (list-member v l): given a value v and a list l, return whether v is a
; member of l. This function works much like the built-in "member"
; predicate, except that it works even when v is a list (e.g., "(not a)").
(defun list-member (v l)
	(cond ((null l) nil)
				((equal v (car l)) t)
				(t (list-member v (cdr l)))))

#| ##### Other Pruning Events ##### |#

; The two negation functions are based on a conversation I had with Sam
; Luckenbill about whether two clauses of the form (a (not b) c) and
; ((not a) b d) can be resolved in two different ways: to (b (not b) c d)
; and (a (not a) c d). We concluded that both of these cases lead to
; vacuous truths, so we throw out both.

; (has-negate l): given a list l, return whether l contains any symbol and
; its negation. It runs recursively on each element in l, running
; has-unit-negate to determine if a negation exists between the element
; in l and the rest of l.
(defun has-negate (l)
	(cond ((null l) nil)
				((has-unit-negate (car l) (cdr l)) t)
				(t (has-negate (cdr l)))))

; (has-unit-negate v l): given a value v and a list l, return whether l
; contains the negation of v. This function runs recursively on l, checking
; if that element is the negation of v.
(defun has-unit-negate (v l)
	(cond ((null l) nil)
				((negate v (car l)) t)
				(t (has-unit-negate v (cdr l)))))

; (drop-big l n): given a list of clause nodes l and a maximum f-cost n,
; return l without any node whose f-cost is greater than n (see prove-aux).
; Because the list is ordered by f-cost already, as soon as we find one
; node with an f-cost that is too high, the rest of the nodes will also
; have too-high f-costs, so we just return nil.
(defun drop-big (l n)
	(cond ((null l) nil) ; no more nodes
				((> (+ (car (car l)) (length (third (car l)))) n) nil) ; too high, we're done
				(t (cons (car l) (drop-big (cdr l) n))))) ; a good node, keep going

#| ##### Sorting ##### |#

; (reverse-car-sort l): given a list l, whose elements are lists with
; numerical first elements, this function returns l sorted in reverse order
; by the car of each element. It is a standard insertion sort using the
; auxiliary function car-insert.
(defun reverse-car-sort (l)
	(cond ((null l) nil)
				(t (car-insert (car l) (reverse-car-sort (cdr l))))))

; (car-insert v l): given a value v and a sorted list l, where v is a list
; whose first element is a number, and where every element of l has the
; same property, insert v into l such that no element ahead of v has
; a greater or equal first element.
(defun car-insert (v l)
	(cond ((null l) (list v))
				((>= (car v) (car (car l))) (cons v l))
				(t (cons (car l) (car-insert v (cdr l))))))

; (add-sort l): given a list of clause nodes l, run a standard insertion
; sort on them, where equivalence is determined by f-cost (see prove-aux).
; It uses the auxiliary function add-insert.
(defun add-sort (l)
	(cond ((null l) nil)
				(t (add-insert (car l) (add-sort (cdr l))))))

; (add-insert v l): given a value v and a sort list l, where v and every
; element of l are clause nodes (see build-ds), insert v into l such that
; no element ahead of v has a smaller or equal f-cost.
(defun add-insert (v l)
	(cond ((null l) (list v))
				((<= (+ (car v) (length (third v))) ; calculate f-cost...
						 (+ (car (car l)) (length (third (car l)))))
				 (cons v l))
				(t (cons (car l) (add-insert v (cdr l))))))

; (merge-nodes l1 l2): given two lists of clause nodes, already sorted by
; lowest f-cost, return a single list with the two lists merged and still
; sorted. The function is the latter half of a standard merge-sort (without
; the initial splitting), and prefers the elements in the second list in
; case of equality. (In context, this means that it prefers to add new
; clauses to the priority queue before old clauses, since the new clauses
; are more likely to be part of a useful resolution chain.)
(defun merge-nodes (l1 l2)
	(cond ((null l1) l2) ; no more of l1 to merge
				((null l2) l1) ; no more of l2 to merge
				((< (+ (car (car l1)) (length (third (car l1)))) ; (car l1) is better
						(+ (car (car l2)) (length (third (car l2)))))
				 (cons (car l1) (merge-nodes (cdr l1) l2))) ; so add it first
				(t (cons (car l2) (merge-nodes l1 (cdr l2)))))) ; else, add (car l2) first
